<?php

// If this file is called directly, abort.
defined('ABSPATH') || die('No direct access!');

class Dragonizer_Database
{
    private $database = [];
    private static $instance;

    public static function getInstance($database_options)
    {
        if (is_null(self::$instance)) {
            self::$instance = new self($database_options);
        }
        return self::$instance;
    }

    public function __construct($database_options)
    {
        $this->database = $database_options;

        add_action('wp_ajax_dragonizer_database_cleanup', array($this, 'clean_up_database'));
        add_action('wp_ajax_dragonizer_database_optimize', array($this, 'optimize_database'));
        add_action('wp_ajax_dragonizer_convert_engine', array($this, 'convert_to_innodb'));

        if (!wp_doing_ajax()) {

            // db_optimizer
            $tag      = $_GET['dragonizer_clean_db'] ?? null;
            $optimize = $_GET['dragonizer_optimize_db'] ?? null;
            $engine   = $_GET['dragonizer_change_engine'] ?? null;

            if (!$tag && !$optimize && !$engine) return;

            if ($tag && check_admin_referer('dragonizer_' . $tag)) {
                $this->clean_up_database($tag);
            } elseif ($optimize && check_admin_referer('dragonizer_optimize_db')) {
                $this->optimize_database(true);
            } elseif ($engine && check_admin_referer('dragonizer_change_engine')) {
                $this->convert_to_innodb(true);
            }

            // db_optimizer
            if ($tag || $optimize || $engine) {
                $url = esc_url(add_query_arg(array('page' => "dragonizer_db_optimizer"), admin_url('admin.php')));
                header("Location: $url");
            }
        }
    }

    public function count_database_bloats($type): int
    {
        global $wpdb;
        switch ($type) {
            case 'autodraft':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `$wpdb->posts` WHERE post_status = %s", 'auto-draft')) ?: 0;

            case 'transient_feed':
                return $wpdb->get_var("SELECT COUNT(*) FROM `$wpdb->options` WHERE option_name LIKE '_site_transient_browser_%' OR option_name LIKE '_site_transient_timeout_browser_%' OR option_name LIKE '_transient_feed_%' OR option_name LIKE '_transient_timeout_feed_%'") ?: 0;

            case 'unapproved_comments':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `$wpdb->comments` WHERE comment_approved = %s", '0')) ?: 0;

            case 'comments_meta':
                return $wpdb->get_var("SELECT COUNT(*) FROM `$wpdb->commentmeta` WHERE comment_id NOT IN (SELECT comment_id FROM `$wpdb->comments`)") ?: 0;

            case 'posts_meta':
                return $wpdb->get_var("SELECT COUNT(*) FROM `$wpdb->postmeta` pm LEFT JOIN `$wpdb->posts` wp ON wp.ID = pm.post_id WHERE wp.ID IS NULL") ?: 0;

            case 'relationships':
                return $wpdb->get_var("SELECT COUNT(*) FROM `$wpdb->term_relationships` WHERE term_taxonomy_id = '1' && object_id NOT IN (SELECT id FROM `$wpdb->posts`)") ?: 0;

            case 'revision':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `$wpdb->posts` WHERE post_type =  %s", 'revision')) ?: 0;

            case 'remove_pingbacks':
                return $wpdb->get_var("SELECT COUNT(*) FROM `$wpdb->comments` WHERE comment_type = 'pingback'") ?: 0;

            case 'remove_trackbacks':
                return $wpdb->get_var("SELECT COUNT(*) FROM `$wpdb->comments` WHERE comment_type = 'trackback'") ?: 0;

            case 'remove_transient_options':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `$wpdb->options` WHERE (option_name LIKE '_transient_timeout_%' && option_value < %d) OR (option_name LIKE '_site_transient_timeout_%' && option_value < %d)", time(), time())) ?: 0;

            case 'spam':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `$wpdb->comments` WHERE comment_approved = %s", 'spam')) ?: 0;

            case 'trash':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `$wpdb->comments` WHERE comment_approved = %s", 'trash')) ?: 0;

            case 'draft':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM `$wpdb->posts` WHERE post_status = %s && (post_type = %s OR post_type = %s)", 'draft', 'page', 'post')) ?: 0;

            case 'deleted_posts':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(ID) FROM `$wpdb->posts` WHERE post_status = %s", 'trash')) ?: 0;

            case 'duplicated_postmeta':
                $query = $wpdb->get_col($wpdb->prepare("SELECT COUNT(meta_id) AS count FROM `$wpdb->postmeta` GROUP BY post_id, meta_key, meta_value HAVING count > %d", 1));
                return is_array($query) ? (array_sum(array_map('intval', $query)) - count($query)) : 0;

            case 'oembed_caches':
                return $wpdb->get_var($wpdb->prepare("SELECT COUNT(meta_id) FROM `$wpdb->postmeta` WHERE meta_key LIKE(%s)", '%_oembed_%')) ?: 0;

            case 'duplicated_commentmeta':
                $query = $wpdb->get_col($wpdb->prepare("SELECT COUNT(meta_id) AS count FROM `$wpdb->commentmeta` GROUP BY comment_id, meta_key, meta_value HAVING count > %d", 1));
                return is_array($query) ? (array_sum(array_map('intval', $query)) - count($query)) : 0;

            case 'orphan_user_meta':
                return $wpdb->get_var("SELECT COUNT(umeta_id) FROM `$wpdb->usermeta` WHERE user_id NOT IN (SELECT ID FROM `$wpdb->users`)") ?: 0;

            case 'duplicated_usermeta':
                $query = $wpdb->get_col($wpdb->prepare("SELECT COUNT(umeta_id) AS count FROM `$wpdb->usermeta`  GROUP BY user_id, meta_key, meta_value HAVING count > %d", 1));
                return is_array($query) ? (array_sum(array_map('intval', $query)) - count($query)) : 0;

                // case 'orphaned_term_relationships':
                //     return $wpdb->get_var("SELECT COUNT(object_id) FROM `$wpdb->term_relationships` AS tr INNER JOIN `$wpdb->term_taxonomy` AS tt ON tr.term_taxonomy_id = tt.term_taxonomy_id WHERE tt.taxonomy != 'link_category' AND tr.object_id NOT IN (SELECT ID FROM `$wpdb->posts`)") ?: 0;

                // case 'unused_terms':
                //     return $wpdb->get_var("");
        }

        return 0;
    }

    public function optimize_database($direct = false)
    {
        global $wpdb;
        $query = 'SHOW TABLE STATUS FROM `' . DB_NAME . '`';
        $fetch = $wpdb->get_results($query);

        // $repair = $this->count_myisam_tables() === 0;

        foreach ($fetch as $row) {
            $wpdb->query("OPTIMIZE TABLE `$row->Name`");
            // $repair && $wpdb->query("REPAIR TABLE `$row->Name`");
        }

        if ($direct) return;

        $this->generate_all_database_fileds();
        wp_die();
    }

    private function get_database_bloat_types()
    {
        $types = ['autodraft', 'transient_feed', 'unapproved_comments', 'comments_meta', 'posts_meta', 'relationships', 'revision', 'remove_pingbacks', 'remove_transient_options', 'remove_trackbacks', 'spam', 'trash', 'draft', 'deleted_posts', 'duplicated_postmeta', 'oembed_caches', 'duplicated_commentmeta', 'orphan_user_meta', 'duplicated_usermeta', 'orphaned_term_relationships', 'unused_terms'];
        foreach ($types as $key => $type) {

            if ($this->count_database_bloats($type) == '0') {
                unset($types[$key]);
            }
        }

        return $types;
    }

    public function clean_up_database($input = false)
    {
        global $wpdb;

        $input = $input ?: $_POST['types'];
        $types = $input != 'all' ? [$input] : $this->get_database_bloat_types();

        start:

        if (empty($types)) goto end;
        if (in_array('autodraft', $types)) {
            $wpdb->delete($wpdb->posts, ['post_status' => 'auto-draft']);
            if (!in_array('posts_meta', $types)) array_push($types, 'posts_meta');
            unset($types['autodraft']);
        }

        if (empty($types)) goto end;
        if (in_array('unapproved_comments', $types)) {
            $wpdb->delete($wpdb->comments, ['comment_approved' => '0']);
            array_push($types, 'comments_meta');
            unset($types['unapproved_comments']);
        }

        if (empty($types)) goto end;
        if (in_array('relationships', $types)) {
            $wpdb->query($wpdb->prepare("DELETE FROM `$wpdb->term_relationships` WHERE term_taxonomy_id=%d AND object_id NOT IN (SELECT id FROM `$wpdb->posts`)", 1));
            unset($types['relationships']);
        }

        if (empty($types)) goto end;
        if (in_array('remove_pingbacks', $types)) {
            $wpdb->delete($wpdb->comments, ['comment_type' => 'pingback']);
            unset($types['remove_pingbacks']);
        }

        if (empty($types)) goto end;
        if (in_array('remove_transient_options', $types)) {
            $wpdb->query($wpdb->prepare("DELETE FROM `$wpdb->options` WHERE (option_name LIKE '_transient_timeout_%' AND option_value < %d) OR (option_name LIKE '_site_transient_timeout_%' AND option_value < %d)", time(), time()));
            // array_push($types, 'transient_feed');
            unset($types['remove_transient_options']);
        }

        if (empty($types)) goto end;
        if (in_array('remove_trackbacks', $types)) {
            $wpdb->delete($wpdb->comments, ['comment_type' => 'trackback']);
            unset($types['remove_trackbacks']);
        }

        if (empty($types)) goto end;
        if (in_array('spam', $types)) {
            $wpdb->delete($wpdb->comments, ['comment_approved' => 'spam']);
            array_push($types, 'comments_meta');
            unset($types['spam']);
        }

        if (empty($types)) goto end;
        if (in_array('trash', $types)) {
            $wpdb->delete($wpdb->comments, ['comment_approved' => 'trash']);
            if (!in_array('posts_meta', $types)) array_push($types, 'posts_meta');
            unset($types['trash']);
        }

        if (empty($types)) goto end;
        if (in_array('draft', $types)) {
            $wpdb->delete($wpdb->posts, ['post_status' => 'draft']);
            if (!in_array('posts_meta', $types)) array_push($types, 'posts_meta');
            unset($types['draft']);
        }

        if (empty($types)) goto end;
        if (in_array('deleted_posts', $types)) {
            $wpdb->delete($wpdb->posts, ['post_status' => 'trash']);
            unset($types['deleted_posts']);
        }

        if (empty($types)) goto end;
        if (in_array('duplicated_postmeta', $types)) {
            if (!function_exists('get_where_sql')) {
                function get_where_sql()
                {
                    global $wpdb;
                    return sprintf("WHERE meta_id NOT IN ( SELECT * FROM ( SELECT MAX(meta_id) FROM `$wpdb->postmeta` GROUP BY post_id, meta_key,meta_value ) AS x )");
                }
            }
            $where_sql = get_where_sql();
            $wpdb->query("DELETE FROM `$wpdb->postmeta` $where_sql");
            unset($types['duplicated_postmeta']);
        }

        if (empty($types)) goto end;
        if (in_array('oembed_caches', $types)) {
            $query = $wpdb->get_results($wpdb->prepare("SELECT post_id, meta_key FROM `$wpdb->postmeta` WHERE meta_key LIKE (%s)", '%_oembed_%'));
            if ($query) {
                foreach ($query as $meta) {
                    $post_id = intval($meta->post_id);
                    if (0 === $post_id) {
                        $wpdb->query($wpdb->prepare("DELETE FROM `$wpdb->postmeta` WHERE post_id = %d AND meta_key = %s", $post_id, $meta->meta_key));
                    } else {
                        delete_post_meta($post_id, $meta->meta_key);
                    }
                }
            }
            unset($types['oembed_caches']);
        }

        if (empty($types)) goto end;
        if (in_array('duplicated_commentmeta', $types)) {
            $query = $wpdb->get_results($wpdb->prepare("SELECT GROUP_CONCAT(meta_id ORDER BY meta_id DESC) AS ids, comment_id, COUNT(*) AS count FROM `$wpdb->commentmeta` GROUP BY comment_id, meta_key, meta_value HAVING count > %d", 1));
            if ($query) {
                foreach ($query as $meta) {
                    $ids = array_map('intval', explode(',', $meta->ids));
                    array_pop($ids);
                    $wpdb->query($wpdb->prepare('DELETE FROM ' . $wpdb->commentmeta . ' WHERE meta_id IN (' . implode(',', $ids) . ') AND comment_id = %d', intval($meta->comment_id)));
                }
            }
            unset($types['duplicated_commentmeta']);
        }

        if (empty($types)) goto end;
        if (in_array('orphan_user_meta', $types)) {
            $wpdb->query("DELETE FROM `$wpdb->usermeta` WHERE user_id NOT IN (SELECT ID FROM `$wpdb->users`)");
            unset($types['orphan_user_meta']);
        }

        if (empty($types)) goto end;
        if (in_array('duplicated_usermeta', $types)) {
            $query = $wpdb->get_results($wpdb->prepare("SELECT GROUP_CONCAT(umeta_id ORDER BY umeta_id DESC) AS ids, user_id, COUNT(*) AS count FROM " . $wpdb->usermeta  . " GROUP BY user_id, meta_key, meta_value HAVING count > %d", 1));
            if ($query) {
                foreach ($query as $meta) {
                    $ids = array_map('intval', explode(',', $meta->ids));
                    array_pop($ids);
                    $wpdb->query($wpdb->prepare('DELETE FROM ' . $wpdb->usermeta . ' WHERE umeta_id IN (' . implode(',', $ids) . ') AND user_id = %d', intval($meta->user_id)));
                }
            }
            unset($types['duplicated_usermeta']);
        }

        if (empty($types)) goto end;
        if (in_array('unused_terms', $types)) {
            unset($types['unused_terms']);
        }

        if (empty($types)) goto end;
        if (in_array('comments_meta', $types)) {
            $wpdb->query("DELETE FROM `$wpdb->commentmeta` WHERE comment_id NOT IN (SELECT comment_id FROM `$wpdb->comments`)");
            unset($types['comments_meta']);
        }

        if (empty($types)) goto end;
        if (in_array('transient_feed', $types)) {
            $wpdb->query($wpdb->prepare("DELETE FROM `$wpdb->options` WHERE option_name LIKE %s OR option_name LIKE %s OR option_name LIKE %s OR option_name LIKE %s", '_site_transient_browser_%', '_site_transient_timeout_browser_%', '_transient_feed_%', '_transient_timeout_feed_%'));
            unset($types['transient_feed']);
        }

        if (empty($types)) goto end;
        if (in_array('revision', $types)) {
            $wpdb->delete($wpdb->posts, ['post_type' => 'revision']);
            array_push($types, 'posts_meta');
            unset($types['revision']);
        }

        if (in_array('posts_meta', $types)) {
            $wpdb->query("DELETE pm FROM `$wpdb->postmeta` pm LEFT JOIN `$wpdb->posts` wp ON wp.ID = pm.post_id WHERE wp.ID IS NULL");
            unset($types['posts_meta']);
        }

        if ($input == 'all') {
            $types = $this->get_database_bloat_types();
            if (!empty($types)) {
                goto start;
            }
        }

        end:
        if (!isset($_POST['types'])) return;

        sleep(1);
        $this->generate_all_database_fileds();
        wp_die();
    }

    private function get_database_size()
    {
        global $wpdb;
        $size = $wpdb->get_var("SELECT SUM(data_length + index_length) FROM information_schema.tables WHERE TABLE_SCHEMA = '" . $wpdb->dbname . "' GROUP BY table_schema");
        return $this->formatBytes($size, 2);
    }

    private function formatBytes($size, $precision = 2)
    {
        $base = log($size, 1024);
        $suffixes = array(__(' B', 'dragonizer'), __(' KB', 'dragonizer'), __(' MB', 'dragonizer'), __(' GB', 'dragonizer'), __(' TB', 'dragonizer'));

        return round(pow(1024, $base - floor($base)), $precision) . ' ' . $suffixes[floor($base)];
    }

    private function count_myisam_tables(): int
    {
        global $wpdb;
        $query = $wpdb->get_results("SHOW TABLE STATUS WHERE Engine = 'MyISAM'", $output = 'ARRAY_A');
        return count($query);
    }

    private function count_innodb_tables(): int
    {
        global $wpdb;
        $query = $wpdb->get_results("SHOW TABLE STATUS WHERE Engine = 'InnoDB'", $output = 'ARRAY_A');
        return count($query);
    }

    public function convert_to_innodb($direct = false): void
    {
        global $wpdb;
        $query = $wpdb->get_results("SHOW TABLE STATUS WHERE Engine = 'MyISAM'", $output = 'ARRAY_A');
        foreach ($query as $table) {
            $table_name = $table['Name'];
            $wpdb->query("ALTER TABLE {$table_name} ENGINE=InnoDB");
        }

        if ($direct) return;

        sleep(1);
        $this->generate_all_database_fileds();
        wp_die();
    }

    public function generate_all_database_fileds()
    {
?>
        <div class="dragonizer_panel_db_optimizer">

            <p class="dragon-hidden"><?= __('<span style="color: #d94f4f">Notice: </span>This section has been programmed with a lot of obsession and all our efforts have been to remove only useless information from your database. <span style="color: #d94f4f"><br>However, we recommend that you back up your database and then optimize it safely.</span>', 'dragonizer') ?></p> <br>
            <h3 class="dragon-hidden"><?php esc_html_e('Database Cleanup', 'dragonizer'); ?> <span class="dragonizer_security_badge dragonizer_danger_badge"><?php esc_html_e('Handle with care', 'dragonizer'); ?></span></h3>
            <p class="dragon-hidden"><?php esc_html_e('This section can help you to keep your database clean by deleting all unnecessary entries.', 'dragonizer'); ?></p> <br>
            <div class="dragonizer_panel-wrapper dragonizer_cards-wrapper">
                <?php
                $total = 0;
                foreach ($this->database as $tag => $v) {
                    if ($tag != 'all') {
                        // Initialize 'count' index if it doesn't exist
                        if (!isset($this->database[$tag]['count'])) {
                            $this->database[$tag]['count'] = 0; // Default value if not set
                        }
                        $this->database[$tag]['count'] = $this->count_database_bloats($tag);
                        if (!in_array($tag, array('optimize_tables'))) {
                            $total += $this->database[$tag]['count'];
                        }
                    }
                    $nonce = wp_create_nonce('dragonizer_' . $tag);
                    // Check the 'count' index directly without using @ to suppress errors
                    $this->database[$tag]['link'] = (empty($this->database[$tag]['count']) || $this->database[$tag]['count'] == '0') ? '#' : esc_url(add_query_arg(array('page' => "dragonizer_db_optimizer", 'dragonizer_clean_db' => $tag, '_wpnonce' => $nonce), admin_url('admin.php')));
                }


                $this->database['all']['count'] = $total;

                foreach ($this->database as $tag => $v) : ?>

                    <a id="<?php echo $tag; ?>" href="<?php echo $v['link']; ?>" class="dragon-hidden dragonizer_panel postbox">
                        <input type="hidden" value="<?= $v['count'] ?>">
                        <section class="dragonizer_panel-wrapper-icon">
                            <span class="dragonizer_panel-icon-<?php echo $tag; ?>"></span>
                        </section>
                        <section class="dragonizer_panel-content">
                            <div class="dragonizer_h3">
                                <?= __($v['title'], 'dragonizer'); ?>
                                <span class="dragonizer_panel-counter<?php if ($v['count'] > 0) echo '-red'; ?>"><?php if ($v['count'] > 0) echo '(' . $v['count'] . ')'; ?></span>
                            </div>
                            <span class="dragonizer_panel-para"><?= __($v['description'], 'dragonizer'); ?></span>
                        </section>
                        <section class="dragonizer_panel-wrapper-top-right">
                            <span class="dragonizer_panel-top-right-icon<?php echo $v['count'] > 0 ? '-cross' : '-tick'; ?>"></span>
                        </section>
                    </a>
                <?php
                endforeach;
                ?>
            </div> <br><br>
            <h3 class="dragon-hidden"><?php esc_html_e('Database Table Engine Converter', 'dragonizer'); ?> <span class="dragonizer_security_badge dragonizer_danger_badge"><?php esc_html_e('Handle with care', 'dragonizer'); ?></span></h3>
            <p class="dragon-hidden"><?php _e('InnoDB is a faster database engine compared to MyISAM, especially when it comes to multitasking. Older sites often use MyISAM as the preferred database engine, and these can be converted to InnoDB instead.', 'dragonizer'); ?></p> <br>

            <?php

            $MyISAM_count = $this->count_myisam_tables();
            $InnoDB_count = $this->count_innodb_tables();

            $nonce = wp_create_nonce('dragonizer_optimize_db');
            $link = $MyISAM_count > 0 ? esc_url(add_query_arg(array('page' => "dragonizer_db_optimizer", 'dragonizer_change_engine' => 'true', '_wpnonce' => $nonce), admin_url('admin.php'))) : '#';

            ?>

            <div class="dragonizer_panel-wrapper dragonizer_cards-wrapper">
                <a style="margin-left: auto; margin-right: auto; width: 50%;" id="convert_engine_button" href="<?= $link ?>" class="dragon-hidden dragonizer_panel postbox">
                    <input type="hidden" value="<?= $MyISAM_count ?>">
                    <section class="dragonizer_panel-wrapper-icon">
                        <span class="dragonizer_panel-icon-convert_db"></span>
                    </section>
                    <section class="dragonizer_panel-content">
                        <div class="dragonizer_h3">
                            <?= __('InnoDB tables', 'dragonizer') ?>
                            <span class="dragonizer_panel-counter">(<?= $InnoDB_count ?>)</span>
                        </div>
                        <br>
                        <div class="dragonizer_h3">
                            <?= __('MyISAM tables', 'dragonizer') ?>
                            <span class="dragonizer_panel-counter<?php if ($MyISAM_count > 0) echo '-red' ?>">(<?= $MyISAM_count ?>)</span>
                        </div>
                        <span <?= $MyISAM_count > 0 ? 'style="color:red"' : 'style="color:green"' ?> class="dragonizer_panel-para"><?= $MyISAM_count > 0 ? __('Click on this section to convert MyISAM tables to InnoDB', 'dragonizer') : __('Your database tables do not need to be converted', 'dragonizer') ?></span>
                    </section>
                    <section class="dragonizer_panel-wrapper-top-right">
                        <span class="dragonizer_panel-top-right-icon-<?= $MyISAM_count > 0 ? 'cross' : 'tick' ?>"></span>
                    </section>
                </a>
            </div> <br><br>
            <h3 class="dragon-hidden"><?php esc_html_e('Database repair and optimization', 'dragonizer'); ?> <span class="dragonizer_security_badge dragonizer_danger_badge"><?php esc_html_e('Handle with care', 'dragonizer'); ?></span></h3>
            <p class="dragon-hidden"><?php _e('This section reorganizes the physical storage of database, which can reduce storage space and improve the database speed.', 'dragonizer'); ?></p> <br>

            <?php

            $nonce = wp_create_nonce('dragonizer_optimize_db');
            $link  = esc_url(add_query_arg(array('page' => "dragonizer_db_optimizer", 'dragonizer_optimize_db' => 'true', '_wpnonce' => $nonce), admin_url('admin.php')));

            ?>
            <div class="dragon-hidden dragonizer_panel-wrapper dragonizer_cards-wrapper">
                <a style="margin-left: auto; margin-right: auto; width: 50%;" id="optimize_database_button" href="<?= $link ?>" class="dragon-hidden dragonizer_panel postbox">
                    <section class="dragonizer_panel-wrapper-icon">
                        <span class="dragonizer_panel-icon-optimize"></span>
                    </section>
                    <section class="dragonizer_panel-content">
                        <div class="dragonizer_h3">
                            <?= __('Database Size: ', 'dragonizer') ?><?= $this->get_database_size() ?>
                        </div>
                        <span style="color:blueviolet" class="dragonizer_panel-para"><?= __('Click on this section to optimize your site database', 'dragonizer') ?></span>
                    </section>
                    <section class="dragonizer_panel-wrapper-top-right">
                        <span class="dragonizer_panel-top-right-icon-star"></span>
                    </section>
                </a>
            </div> <br><br>
            <p class="dragon-hidden"><?= __('<span style="color: #d94f4f">Notice: </span>Database converter engine and database optimization are completely safe, unless the operation is incomplete for reasons such as power outage or internet outage. So we recommend that you backup your database!', 'dragonizer') ?></p>
        </div>
        <br>
<?php }
}
